//***************************************************************************************
//  MSP430 HD44780 LCD demo
//
//  Description; affiche des informations sur un cran LCD
//  1 compteur incrment  chaque appui sur un bouton poussoir
//  1 barregraphe voluant de manire sinusoidale
//  1 chaine de caractres
//  Des caractres personnaliss.

//  ACLK = n/a, MCLK = SMCLK = default DCO
//
//                MSP430x5xx
//             -----------------
//         /|\|              XIN|-
//          | |                 |
//          --|RST          XOUT|-
//            |                 |
//            |             P1.0|-->LCD RS  + LED0
//            |             P1.1|-->LCD EN
//            |             P1.4|-->LCD D4
//            |             P1.5|-->LCD D5
//            |             P1.6|-->LCD D6
//            |             P1.7|-->LCD D7
//            |             P1.3|<--BOUTON
//           c            GND |---GND LCD
//


// Alimentation du LCD par la broche TP1 du launchpad (+5V de l'USB)
// rglage du contraste via un potentiomtre 10KiloOhms
// 5V USB ----
//           |
//          ///
//          ///
//          ///
//          ///
//          ///---->VO LCD
//          ///
//          ///
//           |
// GND    ----
//
//  La plupart des routines du LCD proviennent de http://www.google.fr/url?sa=t&rct=j&q=&esrc=s&source=web&cd=1&cad=rja&ved=0CDUQFjAA&url=http%3A%2F%2Fe2e.ti.com%2Fcfs-file.ashx%2F__key%2FCommunityServer-Discussions-Components-Files%2F166%2F2451.check.txt&ei=ZnwXUo3RGqKn0wXrvoCQBQ&usg=AFQjCNGH6qW7E9pT29wMqjkf0CfnSrSoxg&sig2=iEeNWX8UlLDrtAAHK3RAdA&bvm=bv.51156542,d.d2k
//  programme de dmo par  B. Vandeportaele IUT GEII, Toulouse
//  July 2013
//  Built with Code Composer Studio v5
//***************************************************************************************

//Tlcharger la doc du controleur LCD Alphanumrique hitachi 44780

#include <stdio.h>
#include <math.h>
#include <msp430.h>				

//
// MSP430 LCD Code
//

//inclusion du fichier qui dfinit les broches et les registres du microcontroleur
//#include  "msp430x20x2.h"
#include  "msp430x20x3.h"
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
//dfinition du port utilis pour l'cran LCD, le port1.
// Dans le code, on fera rfrence aux noms plutot qu'aux numros de port
#define     LCD_DIR               P1DIR
#define     LCD_OUT               P1OUT

// Definitions des broches utilises pour l'cran LCD.
// Dans le code, on fera rfrence aux noms plutot qu'aux numros des broches
// Les donnes sont changes avec l'cran LCD par paquets de 4 bits,
// on utilise les broches 4,5,6 et 7  pour le bus de donne
// on utilise les boches 0 et 1 pour le bus de contrle

#define     LCD_PIN_RS            BIT0          // P1.0
#define     LCD_PIN_EN            BIT1          // P1.1
#define     LCD_PIN_D7            BIT7          // P1.7
#define     LCD_PIN_D6            BIT6          // P1.6
#define     LCD_PIN_D5            BIT5          // P1.5
#define     LCD_PIN_D4            BIT4          // P1.4

//dfinition d'un masque pour l'ensemble des broches utilises par l'cran LCD
#define     LCD_PIN_MASK  ((LCD_PIN_RS | LCD_PIN_EN | LCD_PIN_D7 | LCD_PIN_D6 | LCD_PIN_D5 | LCD_PIN_D4))

//dfinition des valeurs de TRUE et FALSE
#define     FALSE                 0
#define     TRUE                  1
//////////////////////////////////////////////////////////////////////////////////////////////////////////////

// Routine Desc:
//
// Cette fonction doit tre appele chaque fois qu'une donne de 4 bits doit tre lu ou crite
// sur le bus de donne de l'cran LCD
//
// Parameters:
//
//     void.
//
// Return
//
//     void.
//
void PulseLcd()
{
    //
    // EN  l'tat bas
    //
    LCD_OUT &= ~LCD_PIN_EN;
    __delay_cycles(200);

    //
    //  EN  l'tat haut
    //
    LCD_OUT |= LCD_PIN_EN;
    __delay_cycles(200);

    //
    //  EN  l'tat bas  nouveau
    //
    LCD_OUT &= (~LCD_PIN_EN);
    __delay_cycles(200);
}
/////////////////////////////////////////////////////////////////////
//
// Routine Desc:
//
// Envoi d'un octet sur le bus de donne de l'cran LCD en 2 paquets de 4 bits
// les 4 bits de poids fort d'abord
//
// Parameters:
//
//    ByteToSend - octet a envoyer
//
//    IsData -   TRUE si l'octet  envoyer est une donne
//               FALSE si l'octet  envoyer est une commande
//
// Return
//
//     void.
//
void SendByte(char ByteToSend, int IsData)
{
    //
    // toutes les broches du LCD  0
    //
    LCD_OUT &= (~LCD_PIN_MASK);
    // envoi des 4 bits de poids fort

    LCD_OUT |= (ByteToSend & 0xF0);

    if (IsData == TRUE)
    {
        LCD_OUT |= LCD_PIN_RS;
    }
    else
    {
        LCD_OUT &= ~LCD_PIN_RS;
    }
    PulseLcd();
    // envoi des 4 bits de poids faible
    LCD_OUT &= (~LCD_PIN_MASK);
    LCD_OUT |= ((ByteToSend & 0x0F) << 4);

    if (IsData == TRUE)
    {
        LCD_OUT |= LCD_PIN_RS;
    }
    else
    {
        LCD_OUT &= ~LCD_PIN_RS;
    }
    PulseLcd();
}
/////////////////////////////////////////////////////////////////////
//
// Routine Desc:
//
// Positionne le curseur sur l'cran LCD
//
// Parameters:
//
//     Row - numro de la ligne (dmarre  zero)
//
//     Col - numro de la colonne (dmarre  zero)
//
// Return
//
//     void.
//
void LcdSetCursorPosition(char Row, char Col)
{
    char address;

    //
    // calcule l'adresse en DDRAN  partir de Row et Col
    //
    if (Row == 0)
    {
        address = 0;
    }
    else
    {
        address = 0x40;
    }

    address |= Col;

    SendByte(0x80 | address, FALSE);
}
/////////////////////////////////////////////////////////////////////
//
// Routine Desc:
//
// Efface l'cran LCD et repositionne le curseur en ligne 0, colonne 0.
//
// Parameters:
//
//    void.
//
// Return
//
//     void.
//
void ClearLcdScreen()
{
    SendByte(0x01, FALSE);
    SendByte(0x02, FALSE);
}
/////////////////////////////////////////////////////////////////////
// Routine Desc:
//
// Initialise l'cran LCD aprs la mise sous tension
//
// Parameters:
//
//    void.
//
// Return
//
//     void.
//
void InitializeLcd(void)
{
    //
    // Rgle la configuration des broches du microcontroleur et les positionne  l'tat bas

    LCD_DIR |= LCD_PIN_MASK;
    LCD_OUT &= ~(LCD_PIN_MASK);


    // attend que le LCD dmarre.
    __delay_cycles(100000);


    // initialise le LCD

    // 1. rglage interface en mode 4 bits

    LCD_OUT &= ~LCD_PIN_RS;
    LCD_OUT &= ~LCD_PIN_EN;

    LCD_OUT = 0x20;
    PulseLcd();
    SendByte(0x28, FALSE);

    // 2. Commande Display on, cursor on, blink cursor

    SendByte(0x0E, FALSE);


    // 3. rglage pour que le curseur se dplace  chaque nouveau caractre envoy  l'cran LCD

    SendByte(0x06, FALSE);
}
/////////////////////////////////////////////////////////////////////
//
// Routine Desc
//
// affiche une chaine de caractres sur l'cran LCD
//
// Parameters:
//
//    Text - la chaine termine par le caractre \0
//
// Returns
	//
//     void.
//
void PrintStr(char *Text)
{
    char *c; //pointeur temporaire pour parcourir la chaine
    c = Text; //initialis sur le premier caractre de la chaine passe en paramtre
//tant que le pointeur est diffrent de null et que le caractre pont n'est pas \0
    while ((c != 0) && ((*c) != 0))
    {
        SendByte(*c, TRUE); //envoi du caractre  l'cran LCD
        c++;  //incrmentation du pointeur
    }
}
/////////////////////////////////////////////////////////////////////
//un exemple de tableau pour redfinir un caractre
char tabvaisseau[8]={0x15,0x15,0x0E,0x4,0x1F,0x0E,0x15,0x15};
/*
10101
10101
01110
00100
11111
01110
10101
10101
*/

//
// Routine Desc
//
// Dfinit un nouveau caractre personnalis
//
// Parameters:
//
//   num: numro du caractre (de 0  7)  dfinir
//   tab: tableau de 8 char, chaque case du tableau dfinit une ligne (via les 5 bits de poids faible) du nouveau caractre
// Returns
//
//     void.
//
void DefineNewCar(unsigned char num, unsigned char * tab)
{
 	int j;
    SendByte(0x40+num*8, FALSE); //set cgram address=num*8 pour la premire ligne du nouveau caractre
	for (j=0;j<8;j++)
	{
		SendByte(tab[j], TRUE);
	}
	SendByte(0x80, FALSE); //set ddram address=0
}
/////////////////////////////////////////////////////////////////////
// Routine Desc
//
// Dfinit  des caractres personnalisables de 0  5 pour avoir un barre graphe
// en binaire les caractres dont toutes les lignes sont identique et valent:
//  00000, 10000, 11000, 11100, 11110, 11111
// Parameters:
//
//     void
// Returns
//
//     void.
//
void DefineNewCarBarre()
{
const char tab[6]={0,0x10,0x18,0x1C,0x1E,0x1F};
int i,j;
	for (i=0;i<6;i++)
	{
		SendByte(0x40+(i*8), FALSE); //set cgram address=i
		for (j=0;j<8;j++)
		{
			SendByte(tab[i], TRUE);
		}
	}
	SendByte(0x80, FALSE); //set ddram address=0
}
/////////////////////////////////////////////////////////////////////
// Routine Desc
//
// Affiche un barregraphe sur 10 caractres, chaque caractre ayant 6 niveaux possibles (de 0  5).
// Parameters:
//
//    val : la valeur  afficher, doit valoir entre 0 et 50 . par exemple pour val=27, la fonction
// affiche 5 carrs complets de 5 pixels de large, 1 carr de 2 pixels de large, puis 4 caractres blancs pour complter
// l'affichage  10 caractres
// Returns
//
//     void.
//
void AfficheBarreGraphe( unsigned char val)
{
int k;
	for (k=0;k<val/5;k++)
		SendByte(5, TRUE);
	SendByte(val%5, TRUE);
	//complte avec des blancs pour avoir 10 caracteres
	for (k=val/5;k<10;k++)
		 SendByte(0, TRUE);
}
/////////////////////////////////////////////////////////////////////

void main(void)
{
	char chaine[30];
	int cpt=0;
	int cpt2=0;
	int cpt3=0;

    WDTCTL = WDTPW + WDTHOLD;             // Stop watchdog timer

    InitializeLcd();
    ClearLcdScreen();

    SendByte(8|4|0|0, FALSE); //envoie directement une commande pour effacer le curseur

    DefineNewCar(7,tabvaisseau); //dfinit de nouveaux caractres personnaliss
    DefineNewCarBarre();
//code de test pour afficher le 8 caractres personnaliss
/*
    SendByte(0, TRUE);
    SendByte(1, TRUE);
    SendByte(2, TRUE);
    SendByte(3, TRUE);
    SendByte(4, TRUE);
    SendByte(5, TRUE);
    SendByte(6, TRUE);
    SendByte(7, TRUE);
        while(1);
*/

    SendByte(7, TRUE);
    SendByte(7, TRUE);
    PrintStr("IUTGEII 2013");
    SendByte(7, TRUE);
    SendByte(7, TRUE);

//Rglage de la broche P1.3 en entre. Il s'agit du bouton poussoir de la carte
//voir http://processors.wiki.ti.com/index.php/Digital_I/O_%28MSP430%29
//configuration de la broche P1.3 en entre, on met le bit correspondant de P1DIR  0 et on laisse les autres inchangs
    P1DIR &=0xf7;
//autorise une rsistance de tirage  l'tat haut (pull up) sur P1.3 . on voit donc un tat haut par dfaut en entre
   	P1REN = 1 <<3;
   	P1OUT |= (1 <<3);
 //configure la broche P1.3 en entre numrique simple
   	P1SEL&= ~(1<<3);
    P1SEL2&= ~(1<<3);

 //boucle sans fin
    while (1)
    {
//affiche le compteur incrment  chaque appui sur le bouton
    	sprintf(chaine, "%d ",cpt);
        LcdSetCursorPosition(1,0);
        PrintStr(chaine);

 //cpt3 est incrment  chaque itration
 //cpt2 est une fonction en sinus de cpt3, de valeur mini 0 et maxi 50
        cpt3++;
        cpt2=25+25*sin(cpt3/10.);
        sprintf(chaine, "%d ",cpt2);
        LcdSetCursorPosition(1,3); //affichage sous forme numrique
        PrintStr(chaine);
        LcdSetCursorPosition(1,5);
        AfficheBarreGraphe(cpt2); //affichage sous forme barregraphe


	 if ((P1IN & 8)==0) // Si le bouton est prss
	 	 {
		 //mthode simpliste: on ralise l'anti rebond mcanique en attendant entre deux lectures et en bloquant
		 //l'excution du programme jusqu' ce que le bouton soit relach
		  __delay_cycles(100);
		  while  ((P1IN & 8)==0);
		  cpt ++; //incrmentation du compteur cpt, avec limitation  la valeur 99 pour afficher sur 2 digits sur le LCD
		  if (cpt>=100)
			  cpt=0;
	 	 }

       }

}

